import { OpenViewService } from "@/utils";
import { AppBase, IParam, ViewDetail, IApp, IOpenViewService, deepCopy, IUIServiceRegister, IDataServiceRegister } from "@core";
import { AppFuncConfig, AppViewConfig } from './config';
import { DataServiceRegister, UIServiceRegister } from "./register";

export class App extends AppBase implements IApp {

    /**
     * 唯一实例
     *
     * @private
     * @static
     * @memberof App
     */
    private static readonly instance = new App();

    /**
     * 获取唯一实例
     *
     * @static
     * @return {*}  {App}
     * @memberof App
     */
    public static getInstance(): App {
        return App.instance;
    }

    /**
     * 获取所有应用功能
     *
     * @return {*}  {IParam[]}
     * @memberof App
     */
    public getAllFuncs(): IParam[] {
        return AppFuncConfig;
    }

    /**
     * 获取打开视图服务
     *
     * @return {*}  {IOpenViewService}
     * @memberof App
     */
    public getOpenViewService(): IOpenViewService {
        return OpenViewService.getInstance();
    }

    /**
     * 获取UI服务
     *
     * @param {string} entityKey 应用实体名小写
     * @param {*} context 应用上下文
     * @return {Promise<any>}
     * @memberof App
     */
    public getUIService(entityKey: string, context?: IParam): Promise<any> {
        return UIServiceRegister.getInstance().getService(entityKey, context);
    }

    /**
     * 获取数据服务
     *
     * @param {string} entityKey 应用实体名小写
     * @param {*} context 应用上下文
     * @return {Promise<any>}
     * @memberof App
     */
    public getDataService(entityKey: string, context?: IParam): Promise<any> {
        return DataServiceRegister.getInstance().getService(entityKey, context);
    }

    /**
     * 获取视图信息
     *
     * @static
     * @param codeName 视图codeName
     * @return {*}  
     */
    public getViewInfo(codeName: string): ViewDetail | undefined {
        return AppViewConfig[codeName] ? (deepCopy(AppViewConfig[codeName]) as ViewDetail) : undefined;;
    }

}